import photoAccessHelper from "@ohos:file.photoAccessHelper";
import camera from "@ohos:multimedia.camera";
import media from "@ohos:multimedia.media";
import type { BusinessError as BusinessError } from "@ohos:base";
import JSON from "@ohos:util.json";
import fileIo from "@ohos:file.fs";
import fs from "@ohos:file.fs";
import Logger from "@normalized:N&&&realauthdemo/src/main/ets/common/Logger&";
import image from "@ohos:multimedia.image";
const TAG: string = 'CameraService';
const uiContext: UIContext | undefined = AppStorage.get('uiContext');
export class Constants {
    // default zoom radio max.
    static readonly ZOOM_RADIO_MAX: number = 6;
    // default zoom radio step.
    static readonly ZOOM_RADIO_MAX_STEP: number = 0.1;
    // default zoom radio step.
    static readonly ZOOM_RADIO_MIN_STEP: number = 0.01;
    // AUDIO_BITRATE.
    static readonly AUDIO_BITRATE: number = 48000;
    // AUDIO_CHANNELS.
    static readonly AUDIO_CHANNELS: number = 2;
    // AUDIO_SAMPLE_RATE.
    static readonly AUDIO_SAMPLE_RATE: number = 48000;
    // VIDEO_BITRATE.
    static readonly VIDEO_BITRATE: number = 512000;
    // VIDEO_FRAME.
    static readonly MAX_VIDEO_FRAME: number = 60;
}
;
export class SliderValue {
    min: number = 1;
    max: number = 6;
    step: number = 0.1;
}
class CLCameraService {
    private context: Context | undefined = undefined;
    private cameraDeviceIndex: number = 0;
    private cameraManager: camera.CameraManager | undefined = undefined;
    private cameras: Array<camera.CameraDevice> | Array<camera.CameraDevice> = [];
    private cameraInput: camera.CameraInput | undefined = undefined;
    private previewOutput: camera.PreviewOutput | undefined = undefined;
    private photoOutput: camera.PhotoOutput | undefined = undefined;
    private videoOutput: camera.VideoOutput | undefined = undefined;
    private avRecorder: media.AVRecorder | undefined = undefined;
    private session: camera.PhotoSession | camera.VideoSession | undefined = undefined;
    private handlePhotoAssetCb: (photoAsset: photoAccessHelper.PhotoAsset) => void = () => {
    };
    private curCameraDevice: camera.CameraDevice | undefined = undefined;
    private isRecording: boolean = false;
    // One of the recommended camera resolutions.
    private photoProfileObj: camera.Profile = {
        format: 2000,
        size: {
            width: 1080,
            height: 1080
        }
    };
    // One of the recommended preview resolutions.
    private previewProfileObj: camera.Profile = {
        format: 1003,
        size: {
            width: 1080,
            height: 1080
        }
    };
    // One of the recommended video resolutions.
    private videoProfileObj: camera.VideoProfile = {
        format: 1003,
        size: {
            width: 1920,
            height: 1080
        },
        frameRateRange: {
            min: 30,
            max: 60
        }
    };
    private curSceneMode: camera.SceneMode = camera.SceneMode.NORMAL_PHOTO;
    constructor() {
    }
    setSavePictureCallback(callback: (photoAsset: photoAccessHelper.PhotoAsset) => void): void {
        this.handlePhotoAssetCb = callback;
    }
    setSceneMode(sceneMode: camera.SceneMode): void {
        this.curSceneMode = sceneMode;
    }
    getSceneMode(): camera.SceneMode {
        return this.curSceneMode;
    }
    getPreviewProfile(cameraOutputCapability: camera.CameraOutputCapability): camera.Profile | undefined {
        let previewProfiles = cameraOutputCapability.previewProfiles;
        // Logger.info( 'previewProfiles🍌🍌🍌🍌' +JSON.stringify(previewProfiles))
        if (previewProfiles.length < 1) {
            return undefined;
        }
        let targetRatio = 1.6;
        let bestProfile = previewProfiles.reduce((best, profile) => {
            const diff = Math.abs(profile.size.width / profile.size.height - targetRatio);
            const bestDiff = Math.abs(best.size.width / best.size.height - targetRatio);
            return diff < bestDiff ? profile : best;
        });
        // return bestProfile
        let index = previewProfiles.findIndex((previewProfile: camera.Profile) => {
            return previewProfile.size.width === this.previewProfileObj.size.width &&
                previewProfile.size.height === this.previewProfileObj.size.height &&
                previewProfile.format === this.previewProfileObj.format;
        });
        if (index === -1) {
            return undefined;
        }
        return previewProfiles[index];
    }
    getPhotoProfile(cameraOutputCapability: camera.CameraOutputCapability): camera.Profile | undefined {
        let photoProfiles = cameraOutputCapability.photoProfiles;
        // Logger.info( 'photoProfiles🍌🍌🍌🍌' +JSON.stringify(photoProfiles))
        if (photoProfiles.length < 1) {
            return undefined;
        }
        let targetRatio = 1.6;
        let bestProfile = photoProfiles.reduce((best, profile) => {
            const diff = Math.abs(profile.size.width / profile.size.height - targetRatio);
            const bestDiff = Math.abs(best.size.width / best.size.height - targetRatio);
            return diff < bestDiff ? profile : best;
        });
        // return bestProfile
        let index = photoProfiles.findIndex((photoProfile: camera.Profile) => {
            return photoProfile.size.width === this.photoProfileObj.size.width &&
                photoProfile.size.height === this.photoProfileObj.size.height &&
                photoProfile.format === this.photoProfileObj.format;
        });
        if (index === -1) {
            return undefined;
        }
        return photoProfiles[index];
    }
    getVideoProfile(cameraOutputCapability: camera.CameraOutputCapability): camera.VideoProfile | undefined {
        let videoProfiles = cameraOutputCapability.videoProfiles;
        if (videoProfiles.length < 1) {
            return undefined;
        }
        for (let i = 0; i < videoProfiles.length; i++) {
            Logger.info(`getVideoProfile: ${JSON.stringify(videoProfiles[i])}`);
        }
        let index = videoProfiles.findIndex((videoProfile: camera.VideoProfile) => {
            return videoProfile.size.width === this.videoProfileObj.size.width &&
                videoProfile.size.height === this.videoProfileObj.size.height &&
                videoProfile.format === this.videoProfileObj.format &&
                videoProfile.frameRateRange.min <= Constants.MAX_VIDEO_FRAME &&
                videoProfile.frameRateRange.max <= Constants.MAX_VIDEO_FRAME;
        });
        if (index === -1) {
            return undefined;
        }
        return videoProfiles[index];
    }
    isSupportedSceneMode(cameraManager: camera.CameraManager, cameraDevice: camera.CameraDevice): boolean {
        let sceneModes = cameraManager.getSupportedSceneModes(cameraDevice);
        if (sceneModes === undefined) {
            return false;
        }
        let index = sceneModes.findIndex((sceneMode: camera.SceneMode) => {
            return sceneMode === this.curSceneMode;
        });
        if (index === -1) {
            return false;
        }
        return true;
    }
    /**
     * Initialize camera function
     * @param surfaceId - Surface ID
     * @param cameraDeviceIndex - Camera device index
     * @returns No return value
     */
    async initCamera(context: Context, surfaceId: string, cameraDeviceIndex: number = 0): Promise<void> {
        Logger.debug(`initCamera cameraDeviceIndex: ${cameraDeviceIndex}`);
        this.cameraDeviceIndex = cameraDeviceIndex;
        this.context = context;
        try {
            await this.releaseCamera();
            // Get Camera Manager instance.
            this.cameraManager = this.getCameraManagerFn();
            if (this.cameraManager === undefined) {
                Logger.error('cameraManager is undefined');
                return;
            }
            // Get support for specified camera device objects.
            this.cameras = this.getSupportedCamerasFn(this.cameraManager);
            if (this.cameras.length < 1 || this.cameras.length < cameraDeviceIndex + 1) {
                return;
            }
            this.curCameraDevice = this.cameras[cameraDeviceIndex];
            let isSupported = this.isSupportedSceneMode(this.cameraManager, this.curCameraDevice);
            if (!isSupported) {
                Logger.error('The current scene mode is not supported.');
                return;
            }
            let cameraOutputCapability = this.cameraManager.getSupportedOutputCapability(this.curCameraDevice, this.curSceneMode);
            let previewProfile = this.getPreviewProfile(cameraOutputCapability);
            if (previewProfile === undefined) {
                Logger.error('The resolution of the current preview stream is not supported.');
                return;
            }
            this.previewProfileObj = previewProfile;
            // Create previewOutput output object.
            this.previewOutput = this.createPreviewOutputFn(this.cameraManager, this.previewProfileObj, surfaceId);
            if (this.previewOutput === undefined) {
                Logger.error('Failed to create the preview stream.');
                return;
            }
            // Monitor preview events.
            this.previewOutputCallBack(this.previewOutput);
            if (this.curSceneMode === camera.SceneMode.NORMAL_PHOTO) {
                let photoProfile = this.getPhotoProfile(cameraOutputCapability);
                if (photoProfile === undefined) {
                    Logger.error('The resolution of the current photo stream is not supported.');
                    return;
                }
                this.photoProfileObj = photoProfile;
                // Create photoOutput output object.
                this.photoOutput = this.createPhotoOutputFn(this.cameraManager, this.photoProfileObj);
                if (this.photoOutput === undefined) {
                    Logger.error('Failed to create the photo stream.');
                    return;
                }
            }
            else if (this.curSceneMode === camera.SceneMode.NORMAL_VIDEO) {
                let videoProfile = this.getVideoProfile(cameraOutputCapability);
                if (videoProfile === undefined) {
                    Logger.error('The resolution of the current video stream is not supported.');
                    return;
                }
                this.videoProfileObj = videoProfile;
                this.avRecorder = await this.createAVRecorder();
                if (this.avRecorder === undefined) {
                    Logger.error('Failed to create the avRecorder.');
                    return;
                }
                await this.prepareAVRecorder();
                let videoSurfaceId = await this.avRecorder.getInputSurface();
                // Create videoOutput output object.
                this.videoOutput = this.createVideoOutputFn(this.cameraManager, this.videoProfileObj, videoSurfaceId);
                if (this.videoOutput === undefined) {
                    Logger.error('Failed to create the video stream.');
                    return;
                }
            }
            // Create cameraInput output object.
            this.cameraInput = this.createCameraInputFn(this.cameraManager, this.curCameraDevice);
            if (this.cameraInput === undefined) {
                Logger.error('Failed to create the camera input.');
                return;
            }
            // Open the camera.
            let isOpenSuccess = await this.cameraInputOpenFn(this.cameraInput);
            if (!isOpenSuccess) {
                Logger.error('Failed to open the camera.');
                return;
            }
            // Camera status callback.
            this.onCameraStatusChange(this.cameraManager);
            // Monitor error events from CameraInput.
            this.onCameraInputChange(this.cameraInput, this.curCameraDevice);
            // Conversation process.
            await this.sessionFlowFn(this.cameraManager, this.cameraInput, this.previewOutput, this.photoOutput, this.videoOutput);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`initCamera fail: ${JSON.stringify(err)}`);
        }
    }
    /**
     * Obtain variable focal length range.
     */
    getZoomRatioRange(): Array<number> {
        let zoomRatioRange: Array<number> = [];
        if (this.session !== undefined) {
            zoomRatioRange = this.session.getZoomRatioRange();
        }
        return zoomRatioRange;
    }
    /**
     * Zoom
     */
    setZoomRatioFn(zoomRatio: number): void {
        Logger.info(`setZoomRatioFn value ${zoomRatio}`);
        // Obtain supported zoom range.
        try {
            let zoomRatioRange = this.getZoomRatioRange();
            Logger.info(`getZoomRatioRange success: ${JSON.stringify(zoomRatioRange)}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`getZoomRatioRange fail: ${JSON.stringify(err)}`);
        }
        try {
            this.session?.setZoomRatio(zoomRatio);
            Logger.info('setZoomRatioFn success');
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`setZoomRatioFn fail: ${JSON.stringify(err)}`);
        }
    }
    /**
     * Trigger a photo with specified parameters.
     */
    async takePicture(): Promise<void> {
        let cameraDeviceIndex = this.cameraDeviceIndex;
        Logger.info('takePicture start');
        let photoSettings: camera.PhotoCaptureSetting = {
            quality: camera.QualityLevel.QUALITY_LEVEL_HIGH,
            mirror: cameraDeviceIndex ? true : false
        };
        await this.photoOutput?.capture(photoSettings);
        Logger.info('takePicture end');
    }
    async saveCameraPhoto(asset: photoAccessHelper.PhotoAsset) {
        Logger.info('saveCameraPhoto');
        try {
            let context = uiContext?.getHostContext();
            let phAccessHelper = photoAccessHelper.getPhotoAccessHelper(context);
            let assetChangeRequest: photoAccessHelper.MediaAssetChangeRequest = new photoAccessHelper.MediaAssetChangeRequest(asset);
            assetChangeRequest.saveCameraPhoto();
            await phAccessHelper.applyChanges(assetChangeRequest);
            Logger.info('apply saveCameraPhoto successfully');
        }
        catch (err) {
            Logger.error(`apply saveCameraPhoto failed with error: ${err.code}, ${err.message}`);
        }
    }
    /**
     * Release the session and its related parameters.
     */
    async releaseCamera(): Promise<void> {
        Logger.info('releaseCamera is called');
        try {
            await this.previewOutput?.release();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`previewOutput release fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.previewOutput = undefined;
        }
        try {
            await this.photoOutput?.release();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`photoOutput release fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.photoOutput = undefined;
        }
        try {
            await this.avRecorder?.release();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`avRecorder release fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.avRecorder = undefined;
        }
        try {
            await this.videoOutput?.release();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`videoOutput release fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.videoOutput = undefined;
        }
        try {
            this.session?.setFlashMode(camera.FlashMode.FLASH_MODE_CLOSE);
            await this.session?.release();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`captureSession release fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.session = undefined;
        }
        try {
            await this.cameraInput?.close();
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`cameraInput close fail: error: ${JSON.stringify(err)}`);
        }
        finally {
            this.cameraInput = undefined;
        }
        this.offCameraStatusChange();
        Logger.info('releaseCamera success');
    }
    /**
     * Get Camera Manager instance.
     */
    getCameraManagerFn(): camera.CameraManager | undefined {
        if (this.cameraManager) {
            return this.cameraManager;
        }
        let cameraManager: camera.CameraManager | undefined = undefined;
        try {
            cameraManager = camera.getCameraManager(this.context);
            Logger.info(`getCameraManager success: ${cameraManager}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`getCameraManager failed: ${JSON.stringify(err)}`);
        }
        return cameraManager;
    }
    /**
     * Get support for specified camera device objects.
     */
    getSupportedCamerasFn(cameraManager: camera.CameraManager): Array<camera.CameraDevice> {
        let supportedCameras: Array<camera.CameraDevice> = [];
        try {
            supportedCameras = cameraManager.getSupportedCameras();
            Logger.info(`getSupportedCameras success: ${this.cameras}, length: ${this.cameras?.length}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`getSupportedCameras failed: ${JSON.stringify(err)}`);
        }
        return supportedCameras;
    }
    /**
     * Create previewOutput output object.
     */
    createPreviewOutputFn(cameraManager: camera.CameraManager, previewProfileObj: camera.Profile, surfaceId: string): camera.PreviewOutput | undefined {
        let previewOutput: camera.PreviewOutput | undefined = undefined;
        try {
            previewOutput = cameraManager.createPreviewOutput(previewProfileObj, surfaceId);
            Logger.info(`createPreviewOutput success: ${previewOutput}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`createPreviewOutput failed: ${JSON.stringify(err)}`);
        }
        return previewOutput;
    }
    /**
     * Create photoOutPut output object.
     */
    createPhotoOutputFn(cameraManager: camera.CameraManager, photoProfileObj: camera.Profile): camera.PhotoOutput | undefined {
        let photoOutput: camera.PhotoOutput | undefined = undefined;
        try {
            photoOutput = cameraManager.createPhotoOutput(photoProfileObj);
            Logger.info(`createPhotoOutputFn success: ${photoOutput}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`createPhotoOutputFn failed: ${JSON.stringify(err)}`);
        }
        return photoOutput;
    }
    /**
     * Create videoOutPut output object.
     */
    createVideoOutputFn(cameraManager: camera.CameraManager, videoProfileObj: camera.VideoProfile, surfaceId: string): camera.VideoOutput | undefined {
        let videoOutput: camera.VideoOutput | undefined = undefined;
        try {
            videoOutput = cameraManager.createVideoOutput(videoProfileObj, surfaceId);
            Logger.info(`createVideoOutputFn success: ${videoOutput}`);
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`createVideoOutputFn failed: ${JSON.stringify(err)}`);
        }
        return videoOutput;
    }
    /**
     * Create cameraInput output object.
     */
    createCameraInputFn(cameraManager: camera.CameraManager, cameraDevice: camera.CameraDevice): camera.CameraInput | undefined {
        Logger.info('createCameraInputFn is called.');
        let cameraInput: camera.CameraInput | undefined = undefined;
        try {
            cameraInput = cameraManager.createCameraInput(cameraDevice);
            Logger.info('createCameraInputFn success');
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`createCameraInputFn failed: ${JSON.stringify(err)}`);
        }
        return cameraInput;
    }
    /**
     * Open the camera.
     */
    async cameraInputOpenFn(cameraInput: camera.CameraInput): Promise<boolean> {
        let isOpenSuccess = false;
        try {
            await cameraInput.open();
            isOpenSuccess = true;
            Logger.info('cameraInput open success');
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`createCameraInput failed : ${JSON.stringify(err)}`);
        }
        return isOpenSuccess;
    }
    /**
     * Conversation process.
     */
    async sessionFlowFn(cameraManager: camera.CameraManager, cameraInput: camera.CameraInput, previewOutput: camera.PreviewOutput, photoOutput: camera.PhotoOutput | undefined, videoOutput: camera.VideoOutput | undefined): Promise<void> {
        try {
            if (this.curSceneMode === camera.SceneMode.NORMAL_PHOTO) {
                this.session = cameraManager.createSession(this.curSceneMode) as camera.PhotoSession;
            }
            else if (this.curSceneMode === camera.SceneMode.NORMAL_VIDEO) {
                this.session = cameraManager.createSession(this.curSceneMode) as camera.VideoSession;
            }
            if (this.session === undefined) {
                return;
            }
            this.onSessionErrorChange(this.session);
            this.session.beginConfig();
            this.session.addInput(cameraInput);
            this.session.addOutput(previewOutput);
            if (this.curSceneMode === camera.SceneMode.NORMAL_PHOTO) {
                if (photoOutput === undefined) {
                    return;
                }
                this.photoOutputCallBack(photoOutput);
                this.session.addOutput(photoOutput);
            }
            else if (this.curSceneMode === camera.SceneMode.NORMAL_VIDEO) {
                if (videoOutput === undefined) {
                    return;
                }
                this.session.addOutput(videoOutput);
            }
            await this.session.commitConfig();
            if (this.curSceneMode === camera.SceneMode.NORMAL_VIDEO) {
                this.setVideoStabilizationFn(this.session as camera.VideoSession, camera.VideoStabilizationMode.MIDDLE);
            }
            this.updateSliderValue();
            this.setFocusMode(camera.FocusMode.FOCUS_MODE_AUTO);
            await this.session.start();
            Logger.info('sessionFlowFn success');
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`sessionFlowFn fail : ${JSON.stringify(err)}`);
        }
    }
    setVideoStabilizationFn(session: camera.VideoSession, videoStabilizationMode: camera.VideoStabilizationMode): void {
        // Check if the specified video stabilization mode is supported.
        let isVideoStabilizationModeSupported: boolean = session.isVideoStabilizationModeSupported(videoStabilizationMode);
        if (isVideoStabilizationModeSupported) {
            session.setVideoStabilizationMode(videoStabilizationMode);
        }
        Logger.info('setVideoStabilizationFn success');
    }
    /**
     * Update slider data.
     */
    updateSliderValue(): void {
        let zoomRatioRange = this.getZoomRatioRange();
        if (zoomRatioRange.length !== 0) {
            let zoomRatioMin = zoomRatioRange[0];
            let zoomRatioMax = zoomRatioRange[1] > Constants.ZOOM_RADIO_MAX ? Constants.ZOOM_RADIO_MAX : zoomRatioRange[1];
            let sliderStep = zoomRatioRange[1] > Constants.ZOOM_RADIO_MAX ? Constants.ZOOM_RADIO_MAX_STEP : Constants.ZOOM_RADIO_MIN_STEP;
            AppStorage.set<SliderValue>('sliderValue', {
                min: zoomRatioMin,
                max: zoomRatioMax,
                step: sliderStep
            });
        }
    }
    /**
     * Monitoring and photography events.
     */
    photoOutputCallBack(photoOutput: camera.PhotoOutput): void {
        try {
            photoOutput.on('captureStartWithInfo', (err: BusinessError, captureStartInfo: camera.CaptureStartInfo): void => {
                Logger.info(`photoOutputCallBack captureStartWithInfo success: ${JSON.stringify(captureStartInfo)}`);
            });
            photoOutput.on('frameShutter', (err: BusinessError, frameShutterInfo: camera.FrameShutterInfo): void => {
                Logger.info(`photoOutputCallBack frameShutter captureId:
          ${frameShutterInfo.captureId}, timestamp: ${frameShutterInfo.timestamp}`);
            });
            photoOutput.on('captureEnd', (err: BusinessError, captureEndInfo: camera.CaptureEndInfo): void => {
                Logger.info(`photoOutputCallBack captureEnd captureId:
          ${captureEndInfo.captureId}, frameCount: ${captureEndInfo.frameCount}`);
            });
            photoOutput.on('error', (data: BusinessError): void => {
                Logger.info(`photoOutPut data: ${JSON.stringify(data)}`);
            });
            photoOutput.on('photoAssetAvailable', (err: BusinessError, photoAsset: photoAccessHelper.PhotoAsset) => {
                Logger.info('photoAssetAvailable begin');
                if (photoAsset === undefined) {
                    Logger.error('photoAsset is undefined');
                    return;
                }
                this.handlePhotoAssetCb(photoAsset);
            });
        }
        catch (err) {
            Logger.error('photoOutputCallBack error');
        }
    }
    /**
     * Monitor preview events.
     */
    previewOutputCallBack(previewOutput: camera.PreviewOutput): void {
        Logger.info('previewOutputCallBack is called');
        try {
            previewOutput.on('frameStart', (): void => {
                Logger.debug('Preview frame started');
            });
            previewOutput.on('frameEnd', (): void => {
                Logger.debug('Preview frame ended');
            });
            previewOutput.on('error', (previewOutputError: BusinessError): void => {
                Logger.info(`Preview output previewOutputError: ${JSON.stringify(previewOutputError)}`);
            });
        }
        catch (err) {
            Logger.error('previewOutputCallBack error');
        }
    }
    /**
     * Callback function for registering camera status changes.
     * @param err
     * @param cameraStatusInfo
     * @returns
     */
    registerCameraStatusChange(err: BusinessError, cameraStatusInfo: camera.CameraStatusInfo): void {
        Logger.info(`cameraId: ${cameraStatusInfo.camera.cameraId},status: ${cameraStatusInfo.status}`);
    }
    /**
     * Monitor camera status changes.
     * @param cameraManager
     * @returns
     */
    onCameraStatusChange(cameraManager: camera.CameraManager): void {
        Logger.info('onCameraStatusChange is called');
        try {
            cameraManager.on('cameraStatus', this.registerCameraStatusChange);
        }
        catch (error) {
            Logger.error('onCameraStatusChange error');
        }
    }
    /**
     * Stop monitoring camera status changes.
     * @returns
     */
    offCameraStatusChange(): void {
        Logger.info('offCameraStatusChange is called');
        this.cameraManager?.off('cameraStatus', this.registerCameraStatusChange);
    }
    /**
     * Monitor camera input changes.
     * @param cameraInput
     * @param cameraDevice
     * @returns
     */
    onCameraInputChange(cameraInput: camera.CameraInput, cameraDevice: camera.CameraDevice): void {
        Logger.info(`onCameraInputChange is called`);
        try {
            cameraInput.on('error', cameraDevice, (cameraInputError: BusinessError): void => {
                Logger.info(`onCameraInputChange cameraInput error code: ${cameraInputError.code}`);
            });
        }
        catch (error) {
            Logger.error('onCameraInputChange error');
        }
    }
    /**
     * @param session
     * @returns
     */
    onSessionErrorChange(session: camera.PhotoSession | camera.VideoSession): void {
        try {
            session.on('error', (captureSessionError: BusinessError): void => {
                Logger.info('onCaptureSessionErrorChange captureSession fail: ' + JSON.stringify(captureSessionError.code));
            });
        }
        catch (error) {
            Logger.error('onCaptureSessionErrorChange error');
        }
    }
    async createAVRecorder(): Promise<media.AVRecorder | undefined> {
        let avRecorder: media.AVRecorder | undefined = undefined;
        try {
            avRecorder = await media.createAVRecorder();
        }
        catch (error) {
            Logger.error(`createAVRecorder error: ${error}`);
        }
        return avRecorder;
    }
    initFd(): number {
        Logger.info('initFd is called');
        let filesDir = this.context?.filesDir;
        let filePath = filesDir + `/${Date.now()}.mp4`;
        console.log(`输出当前的视频路径${filePath}`);
        AppStorage.setOrCreate<string>('filePath', filePath);
        let file: fileIo.File = fileIo.openSync(filePath, fileIo.OpenMode.READ_WRITE | fileIo.OpenMode.CREATE);
        return file.fd;
    }
    async prepareAVRecorder(): Promise<void> {
        Logger.info('prepareAVRecorder is called');
        let fd = this.initFd();
        let videoConfig: media.AVRecorderConfig = {
            audioSourceType: media.AudioSourceType.AUDIO_SOURCE_TYPE_MIC,
            videoSourceType: media.VideoSourceType.VIDEO_SOURCE_TYPE_SURFACE_YUV,
            profile: {
                audioBitrate: Constants.AUDIO_BITRATE,
                audioChannels: Constants.AUDIO_CHANNELS,
                audioCodec: media.CodecMimeType.AUDIO_AAC,
                audioSampleRate: Constants.AUDIO_SAMPLE_RATE,
                fileFormat: media.ContainerFormatType.CFT_MPEG_4,
                videoBitrate: Constants.VIDEO_BITRATE,
                videoCodec: media.CodecMimeType.VIDEO_AVC,
                videoFrameWidth: this.videoProfileObj.size.width,
                videoFrameHeight: this.videoProfileObj.size.height,
                videoFrameRate: this.videoProfileObj.frameRateRange.max
            },
            url: `fd://${fd.toString()}`,
            rotation: this.curCameraDevice?.cameraOrientation
        };
        Logger.info(`prepareAVRecorder videoConfig: ${JSON.stringify(videoConfig)}`);
        await this.avRecorder?.prepare(videoConfig).catch((err: BusinessError): void => {
            Logger.error(`prepareAVRecorder prepare err: ${JSON.stringify(err)}`);
        });
    }
    async startVideo(): Promise<void> {
        Logger.info('startVideo is called');
        try {
            await this.videoOutput?.start();
            await this.avRecorder?.start();
            this.isRecording = true;
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`startVideo err: ${JSON.stringify(err)}`);
        }
        Logger.info('startVideo End of call');
    }
    async stopVideo(): Promise<void> {
        Logger.info('stopVideo is called');
        if (!this.isRecording) {
            Logger.info('not in recording');
            return;
        }
        try {
            if (this.avRecorder) {
                await this.avRecorder.stop();
            }
            if (this.videoOutput) {
                await this.videoOutput.stop();
            }
            this.isRecording = false;
        }
        catch (error) {
            let err = error as BusinessError;
            Logger.error(`stopVideo err: ${JSON.stringify(err)}`);
        }
        Logger.info('stopVideo End of call');
    }
    /**
     * Flashing lights
     */
    hasFlashFn(flashMode: camera.FlashMode): void {
        let hasFlash = this.session?.hasFlash();
        Logger.debug(`hasFlash success, hasFlash: ${hasFlash}`);
        let isFlashModeSupported = this.session?.isFlashModeSupported(flashMode);
        Logger.debug(`isFlashModeSupported success, isFlashModeSupported: ${isFlashModeSupported}`);
        this.session?.setFlashMode(flashMode);
    }
    getFlashMode(): camera.FlashMode | undefined {
        return this.session?.getFlashMode();
    }
    setFlashMode() {
        if (this.session?.getFlashMode() === camera.FlashMode.FLASH_MODE_OPEN || this.session?.getFlashMode() === camera.FlashMode.FLASH_MODE_ALWAYS_OPEN) {
            this.hasFlashFn(camera.FlashMode.FLASH_MODE_CLOSE);
        }
        else {
            this.hasFlashFn(camera.FlashMode.FLASH_MODE_ALWAYS_OPEN);
        }
    }
    /**
     * Set the current focus
     */
    setFocusPoint(point: camera.Point): void {
        this.session?.setFocusPoint(point);
        Logger.info(`setFocusPoint success point: ${JSON.stringify(point)}`);
        // Get the current focus
        let nowPoint: camera.Point | undefined = undefined;
        nowPoint = this.session?.getFocusPoint();
        Logger.info(`getFocusPoint success, nowPoint: ${JSON.stringify(nowPoint)}`);
    }
    /**
     * Exposure area
     */
    isMeteringPoint(point: camera.Point): void {
        let exposureMode: camera.ExposureMode | undefined = undefined;
        exposureMode = this.session?.getExposureMode();
        Logger.info(`getExposureMode success, exposureMode: ${exposureMode}`);
        this.session?.setMeteringPoint(point);
        let exposurePoint: camera.Point | undefined = undefined;
        exposurePoint = this.session?.getMeteringPoint();
        Logger.info(`getMeteringPoint exposurePoint: ${JSON.stringify(exposurePoint)}`);
    }
    /**
     * Exposure compensation
     */
    isExposureBiasRange(exposureBias: number): void {
        Logger.debug(`setExposureBias value ${exposureBias}`);
        let biasRangeArray: Array<number> | undefined = [];
        biasRangeArray = this.session?.getExposureBiasRange();
        Logger.debug(`getExposureBiasRange success, biasRangeArray: ${JSON.stringify(biasRangeArray)}`);
        this.session?.setExposureBias(exposureBias);
    }
    setFocusMode(focusMode: camera.FocusMode): void {
        Logger.info(`setFocusMode is called`);
        let isSupported = this.session?.isFocusModeSupported(focusMode);
        Logger.info(`setFocusMode isSupported: ${isSupported}`);
        if (!isSupported) {
            return;
        }
        this.session?.setFocusMode(focusMode);
    }
    async getImageUint8Array(uri: string): Promise<Uint8Array> {
        const file = await fs.open(uri, fs.OpenMode.READ_ONLY);
        const imageSource: image.ImageSource = image.createImageSource(file.fd);
        const imagePackerApi = image.createImagePacker();
        // 设置打包参数
        // format：当前仅支持打包为JPEG、WebP 和 png 格式
        // quality：JPEG 编码输出图片质量
        // bufferSize：图片大小，默认 10M
        const packOpts: image.PackingOption = { format: "image/jpeg", quality: 100 };
        let imageBuffer: ArrayBuffer = new ArrayBuffer(1);
        try {
            // 图片压缩或重新打包
            imageBuffer = await imagePackerApi.packing(imageSource, packOpts);
            // let base64Str = buffer.from(imageBuffer).toString('base64')
            // resultBase64Str="data:image/jpeg;base64,"+base64Str
        }
        catch (err) {
            console.error(`Invoke getImageArrayBufferWithUri failed, err: ${JSON.stringify(err)}`);
        }
        return new Uint8Array(imageBuffer);
    }
    openPhotoAlbum(call: (imageURL: string, error?: Error) => void) {
        try {
            let photoSelectOptions = new photoAccessHelper.PhotoSelectOptions();
            photoSelectOptions.MIMEType = photoAccessHelper.PhotoViewMIMETypes.IMAGE_TYPE;
            photoSelectOptions.maxSelectNumber = 1;
            let photoPicker = new photoAccessHelper.PhotoViewPicker();
            photoPicker.select(photoSelectOptions)
                .then((PhotoSelectResult: photoAccessHelper.PhotoSelectResult) => {
                let imageURI = PhotoSelectResult.photoUris[0];
                call(imageURI);
            });
        }
        catch (error) {
            call('', error);
        }
    }
}
export default new CLCameraService();
